//
//  TodoListChooser.swift
//  Do It
//
//  Created by Jim Dovey on 1/27/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

struct Home: View {
    // START:FetchRequest
    @FetchRequest<TodoItemList>(ascending: \.manualSortOrder)
    var lists
    // END:FetchRequest

    // START:NewList
    @Environment(\.managedObjectContext) var objectContext
    // END:NewList
    @State private var showingEditor = false

    // START:ShowEditor
    var body: some View {
        NavigationView {
            // <literal:elide> ... </literal:elide>
            // END:ShowEditor
            List {
                Section(header: HomeHeader().padding(.vertical)) {
                    // START:UpdateListContent
                    // START_HIGHLIGHT
                    ForEach(self.lists) { list in
                    // END_HIGHLIGHT
                        NavigationLink(destination: TodoList(list: list)) {
                            // START_HIGHLIGHT
                            Row(name: list.name ?? "<Unknown>",
                                icon: list.icon ?? "list.bullet",
                                // END_HIGHLIGHT
                                color: list.color.uiColor)
                        }
                    }
                    // END:UpdateListContent
                    // START:OnDelete
                    .onDelete { offsets in
                        self.objectContext.perform {
                            for offset in offsets {
                                self.objectContext.delete(self.lists[offset])
                            }
                            try? self.objectContext.save()
                        }
                    }
                    // END:OnDelete
                    // START:OnMove
                    .onMove { offsets, index in
                        var newOrder = Array(self.lists)
                        newOrder.move(fromOffsets: offsets, toOffset: index)
                        self.objectContext.perform {
                            for (index, list) in newOrder.enumerated() {
                                list.manualSortOrder = Int32(index)
                            }
                            try? self.objectContext.save()
                        }
                    }
                    // END:OnMove
                }
            }
            .font(.system(.headline, design: .rounded))
            .listStyle(GroupedListStyle())
            .navigationBarTitle("Lists")
            .navigationBarItems(leading: EditButton(), trailing: addButton)
            // START:ShowEditor
        }
        // START_HIGHLIGHT
        .sheet(isPresented: $showingEditor,
               onDismiss: { try? self.objectContext.save() },
               content: { self.newListEditor })
        // END_HIGHLIGHT
    }
    // END:ShowEditor
    
    private var addButton: some View {
        Button(action: { self.showingEditor.toggle() }) {
            Image(systemName: "plus.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Add New List"))
        }
    }
    // START:ShowEditor

    private var newListEditor: some View {
        let context = objectContext.editingContext()
        let list = TodoItemList.newList(in: context)
        return TodoListEditor(list: list)
            .environment(\.managedObjectContext, context)
    }
    // END:ShowEditor

    private struct Row: View {
        var name: String
        var icon: String
        var color: Color

        var body: some View {
            HStack {
                Image(systemName: icon)
                    .foregroundColor(.white)
                    .frame(width: 32, height: 32)
                    .background(color)
                    .clipShape(Circle())
                Text(LocalizedStringKey(name))
            }
        }
    }
}

struct TodoListChooser_Previews: PreviewProvider {
    // START:Preview
    static var previews: some View {
        Home()
            .environment(\.managedObjectContext,
                         PreviewDataStore.shared.viewContext)
    }
    // END:Preview
}
